﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System.Collections.Generic;

using GE.Tools;

namespace GE.Message
{
    /// <summary>
    /// Message System Component.
    /// </summary>
    class MessageSystem
    {
        /// <summary>
        /// Unique instance of the MessageSystem class.
        /// </summary>
        private static MessageSystem m_instance = new MessageSystem ();

        /// <summary>
        /// Maps message indices to messages.
        /// </summary>
        private Dictionary<int, Message> m_messages;

        /// <summary>
        /// Next message index.
        /// </summary>
        private int m_index;

        /// <summary>
        /// Unique instance of the MessageSystem class.
        /// </summary>
        public static MessageSystem Instance
        {
            get { return m_instance; }
        }

        /// <summary>
        /// Private constructor.
        /// </summary>
        private MessageSystem()
        {
            m_messages = new Dictionary<int, Message>();
            m_index = 0;
        }

        /// <summary>
        /// Get a list of message matching a specific receiver.
        /// </summary>
        /// <param name="receiver">receiver</param>
        /// <returns>message list</returns>
        public List<Message> getList(MessageReceiver receiver)
        {
            List<Message> messageList = new List<Message>();
            foreach (Message m in m_messages.Values)
            {
                if (m.Receiver == receiver)
                    messageList.Add(m);
            }
            return messageList;
        }

        /// <summary>
        /// Push a message into the message system.
        /// </summary>
        /// <param name="type">message type</param>
        /// <param name="receiver">reciever</param>
        /// <param name="a">field A</param>
        /// <param name="b">field B</param>
        /// <param name="c">field C</param>
        /// <param name="d">field D</param>
        public void pushMessage(MessageType type, MessageReceiver receiver, float a, float b, float c, float d)
        {
            Message m = new Message(m_index, type, receiver);
            m_messages[m_index] = (m);
            m_index++;
            m.A = a;
            m.B = b;
            m.C = c;
            m.D = d;
        }

        /// <summary>
        /// Pop a message from the message system.
        /// </summary>
        /// <param name="index">message index</param>
        public void popMessage(int index)
        {
            try
            {
                m_messages.Remove(index);
            }
            catch (KeyNotFoundException e)
            {
                Logger.Instance.error(this, e, "Message index doesn't exist");
            }
        }

        /// <summary>
        /// Clear the message system.
        /// </summary>
        public void clear()
        {
            m_messages.Clear();
        }
    }
}
